
//
//  pe_mtl_blending.metal
//  PTPaintTools
//
//  Created by Georgy Ostrobrod on 26/06/2017.
//  Copyright © 2017 Pixelmator. All rights reserved.
//

#include <metal_stdlib>
using namespace metal;


// ------------------------------  Blend ---------------------------------------

typedef enum BlendModesEnum {
    BLEND_NORMAL,
    BLEND_DARKEN,
    BLEND_LIGHTEN,
    BLEND_MULTIPLY,
    BLEND_SCREEN,
    BLEND_COLORBURN,
    BLEND_LINEARBURN,
    BLEND_COLORDODGE,
    BLEND_LINEARDODGE,
    BLEND_OVERLAY,
    BLEND_SOFTLIGHT,
    BLEND_HARDLIGHT,
    BLEND_VIVIDLIGHT,
    BLEND_LINEARLIGHT,
    BLEND_PINLIGHT,
    BLEND_DIFFERENCE,
    BLEND_EXCLUSION,
    BLEND_LIGHTERCOLOR,
    BLEND_DARKERCOLOR,
    BLEND_LUMINOSITY,
    BLEND_SATURATE,
    BLEND_HUE,
    BLEND_COLOR,
    BLEND_DIVIDE,
    BLEND_SUBTRACT,
    BLEND_DISSOLVE,
    BLEND_HARDMIX,
    BLEND_BEHIND,
    
    BLEND_COUNT
} BlendModes;


struct BlendParams {
    int alpha_index;
    float2 cur_position;
};



class CBlendNormal {
public:
    template<typename T>
    inline static vec<T, 4> blend(vec<T, 4> src, vec<T, 4> dst, BlendParams params) {
        return src + dst * (1.0 - src[params.alpha_index]);
    }
};



class CBlendDarken {
public:
    
    template<typename T>
    inline static vec<T, 4> blend(vec<T, 4> src, vec<T, 4> dst, BlendParams params) {
        return vec<T, 4>(min(src.rgb * dst[params.alpha_index], dst.rgb * src[params.alpha_index]) +
                         src.rgb * (1.0 - dst[params.alpha_index]) + dst.rgb * (1.0 - src[params.alpha_index]),
                         src[params.alpha_index] + dst[params.alpha_index] * (1.0 - src[params.alpha_index]));
    }
};



class CBlendLighten {
public:
    
    template<typename T>
    inline static vec<T, 4> blend(vec<T, 4> src, vec<T, 4> dst, BlendParams params) {
        return vec<T, 4>(max(src.rgb * dst[params.alpha_index], dst.rgb * src[params.alpha_index]) +
                         src.rgb * (1.0 - dst[params.alpha_index]) + dst.rgb * (1.0 - src[params.alpha_index]),
                         src[params.alpha_index] + dst[params.alpha_index] * (1.0 - src[params.alpha_index]));
    }
};



class CBlendMultiply {
public:
    
    template<typename T>
    inline static vec<T, 4> blend(vec<T, 4> src, vec<T, 4> dst, BlendParams params) {
        return vec<T, 4>(src.rgb * dst.rgb + src.rgb * (1.0 - dst[params.alpha_index]) +
                         dst.rgb * (1.0 - src[params.alpha_index]),
                         src[params.alpha_index] + dst[params.alpha_index] * (1.0 - src[params.alpha_index]));
    }
};



class CBlendScreen {
public:
    
    template<typename T>
    inline static vec<T, 4> blend(vec<T, 4> src, vec<T, 4> dst, BlendParams params) {
        return vec<T, 4>(src.rgb + dst.rgb - src.rgb * dst.rgb,
                         src[params.alpha_index] + dst[params.alpha_index] * (1.0 - src[params.alpha_index]));
    }
};



class CBlendColorBurn {
public:
    
    template<typename T>
    inline static vec<T, 4> blend(vec<T, 4> src, vec<T, 4> dst, BlendParams params) {
        return vec<T, 4>(src[params.alpha_index] * dst[params.alpha_index] * (1.0 - min(src[params.alpha_index] * (dst[params.alpha_index] - dst.rgb) / max(src.rgb * dst[params.alpha_index], 0.001), 1.0)) +
                         src.rgb * (1.0 - dst[params.alpha_index]) + dst.rgb * (1.0 - src[params.alpha_index]),
                         src[params.alpha_index] + dst[params.alpha_index] * (1.0 - src[params.alpha_index]));
    }
};



class CBlendLinearBurn {
public:
    
    template<typename T>
    inline static vec<T, 4> blend(vec<T, 4> src, vec<T, 4> dst, BlendParams params) {
        return vec<T, 4>(src.rgb * dst[params.alpha_index] + dst.rgb * src[params.alpha_index] - dst[params.alpha_index] * src[params.alpha_index] +
                         src.rgb * (1.0 - dst[params.alpha_index]) + dst.rgb * (1.0 - src[params.alpha_index]),
                         src[params.alpha_index] + dst[params.alpha_index] * (1.0 - src[params.alpha_index]));
    }
};



class CBlendColorDodge {
public:
    
    template<typename T>
    inline static vec<T, 4> blend(vec<T, 4> src, vec<T, 4> dst, BlendParams params) {
        return vec<T, 4>(src[params.alpha_index] * dst[params.alpha_index] * min(dst.rgb * src[params.alpha_index] / max(dst[params.alpha_index] * (src[params.alpha_index] - src.rgb), 0.001), 1.0) +
                         src.rgb * (1.0 - dst[params.alpha_index]) + dst.rgb * (1.0 - src[params.alpha_index]),
                         src[params.alpha_index] + dst[params.alpha_index] * (1.0 - src[params.alpha_index]));
    }
};



class CBlendLinearDodge {
public:
    
    template<typename T>
    inline static vec<T, 4> blend(vec<T, 4> src, vec<T, 4> dst, BlendParams params) {
        return vec<T, 4>(src.rgb * dst[params.alpha_index] + dst.rgb * src[params.alpha_index] +
                         src.rgb * (1.0 - dst[params.alpha_index]) + dst.rgb * (1.0 - src[params.alpha_index]),
                         src[params.alpha_index] + dst[params.alpha_index] * (1.0 - src[params.alpha_index]));
    }
};



class CBlendOverlay {
public:
    
    template<typename T>
    inline static vec<T, 4> blend(vec<T, 4> src, vec<T, 4> dst, BlendParams params) {
        vec<T, 3> dst_f = vec<T, 3>(dst.rgb * 2.0 > dst[params.alpha_index]);
        return vec<T, 4>((src.rgb * (1.0 + dst[params.alpha_index]) + dst.rgb * (1.0 + src[params.alpha_index]) -
                          2.0 * dst.rgb * src.rgb - dst[params.alpha_index] * src[params.alpha_index]) * dst_f +
                         (2.0 * src.rgb * dst.rgb + src.rgb * (1.0 - dst[params.alpha_index]) +
                          dst.rgb * (1.0 - src[params.alpha_index])) * (1.0 - dst_f),
                         src[params.alpha_index] + dst[params.alpha_index] * (1.0 - src[params.alpha_index]));
    }
};



class CBlendSoftLight {
public:
    
    template<typename T>
    inline static vec<T, 4> blend(vec<T, 4> src, vec<T, 4> dst, BlendParams params) {
        vec<T,3> m = dst.rgb / max(dst.a, T(0.001));
        vec<T,3> a = 2.0 * src.rgb - src.a;
        vec<T,3> b = dst.a * a;
        vec<T,3> c = src.rgb - src.rgb * dst.a + dst.rgb;
        vec<T,3> d_src = 2.0 * src.rgb;
        vec<T,3> q_dst = 4.0 * dst.rgb;
        vec<T,3> eq1 = vec<T,3>(T(d_src.r <= src.a),
                                T(d_src.g <= src.a),
                                T(d_src.b <= src.a));
        vec<T,3> eq2 = vec<T,3>(T(q_dst.r <= dst.a),
                                T(q_dst.g <= dst.a),
                                T(q_dst.b <= dst.a));
        
        vec<T,3> d = eq2 * (16.0 * m * m * m - 12.0 * m * m + 3.0 * m) + (1.0 - eq2) * (sqrt(m) - m);
        
        vec<T,3> eq1val = dst.rgb * (src.a + a * (1.0 - m)) + src.rgb * (1.0 - dst.a) + dst.rgb * (1.0 - src.a);
        vec<T,3> eq2val = b * d + c;
        
        vec<T,4> res = vec<T,4>(eq1 * eq1val + (1.0 - eq1) * eq2val,
                                src.a + dst.a * (1.0 - src.a));
        return res;
    }
};



class CBlendHardLight {
public:
    
    template<typename T>
    inline static vec<T, 4> blend(vec<T, 4> src, vec<T, 4> dst, BlendParams params) {
        vec<T, 3> dst_f = vec<T, 3>(src.rgb * 2.0 > dst[params.alpha_index]);
        return vec<T, 4>((src.rgb * (1.0 + dst[params.alpha_index]) + dst.rgb * (1.0 + src[params.alpha_index]) -
                          2.0 * dst.rgb * src.rgb - dst[params.alpha_index] * src[params.alpha_index]) * dst_f +
                         (2.0 * src.rgb * dst.rgb + src.rgb * (1.0 - dst[params.alpha_index]) +
                          dst.rgb * (1.0 - src[params.alpha_index])) * (1.0 - dst_f),
                         src[params.alpha_index] + dst[params.alpha_index] * (1.0 - src[params.alpha_index]));
    }
};



class CBlendVividLight {
public:
    
    template<typename T>
    inline static vec<T, 4> blend(vec<T, 4> src, vec<T, 4> dst, BlendParams params) {
        vec<T, 3> dst_f = vec<T, 3>(src.rgb * 2.0 > dst[params.alpha_index]);
        return vec<T, 4>((dst_f * clamp(dst.rgb * src[params.alpha_index] / max(2.0 * (src[params.alpha_index] - src.rgb) * dst[params.alpha_index], T(0.01)), T(0.0), T(1.0)) +
                          (1.0 - dst_f) * clamp((2.0 * dst[params.alpha_index] * src.rgb - (dst[params.alpha_index] - dst.rgb) * src[params.alpha_index]) / max(2.0 * src.rgb * dst[params.alpha_index], T(0.01)), T(0.0), T(1.0)))
                         * src[params.alpha_index] * dst[params.alpha_index] + src.rgb * (1.0 - dst[params.alpha_index]) + dst.rgb * (1.0 - src[params.alpha_index]),
                         src[params.alpha_index] + dst[params.alpha_index] * (1.0 - src[params.alpha_index]));
    }
};



class CBlendLinearLight {
public:
    
    template<typename T>
    inline static vec<T, 4> blend(vec<T, 4> src, vec<T, 4> dst, BlendParams params) {
        return vec<T, 4>(dst.rgb * src[params.alpha_index] + 2.0 * src.rgb * dst[params.alpha_index] - src[params.alpha_index] * dst[params.alpha_index] +
                         src.rgb * (1.0 - dst[params.alpha_index]) + dst.rgb * (1.0 - src[params.alpha_index]),
                         src[params.alpha_index] + dst[params.alpha_index] * (1.0 - src[params.alpha_index]));
    }
};



class CBlendPinLight {
public:
    
    template<typename T>
    inline static vec<T, 4> blend(vec<T, 4> src, vec<T, 4> dst, BlendParams params) {
        vec<T, 3> dst_f = vec<T, 3>(src.rgb * 2.0 > dst[params.alpha_index]);
        return vec<T, 4>(dst_f * clamp(max(dst.rgb * src[params.alpha_index], 2.0 * src.rgb * dst[params.alpha_index] - src[params.alpha_index] * dst[params.alpha_index]), T(0.0), T(1.0)) +
                         (1.0 - dst_f) * clamp(min(dst.rgb * src[params.alpha_index], 2.0 * src.rgb * dst[params.alpha_index]), T(0.0), T(1.0)) +
                         src.rgb * (1.0 - dst[params.alpha_index]) + dst.rgb * (1.0 - src[params.alpha_index]),
                         src[params.alpha_index] + dst[params.alpha_index] * (1.0 - src[params.alpha_index]));
    }
};



class CBlendDifference {
public:
    
    template<typename T>
    inline static vec<T, 4> blend(vec<T, 4> src, vec<T, 4> dst, BlendParams params) {
        return vec<T, 4>(src.rgb + dst.rgb -
                         2.0 * min(src.rgb * dst[params.alpha_index], dst.rgb * src[params.alpha_index]),
                         src[params.alpha_index] + dst[params.alpha_index] * (1.0 - src[params.alpha_index]));
    }
};



class CBlendExclusion {
public:
    
    template<typename T>
    inline static vec<T, 4> blend(vec<T, 4> src, vec<T, 4> dst, BlendParams params) {
        return vec<T, 4>((src.rgb * dst[params.alpha_index] + dst.rgb * src[params.alpha_index] -
                          2.0 * src.rgb * dst.rgb) + src.rgb * (1.0 - dst[params.alpha_index]) +
                         dst.rgb * (1.0 - src[params.alpha_index]),
                         src[params.alpha_index] + dst[params.alpha_index] * (1.0 - src[params.alpha_index]));
    }
};



class CBlendLighterColor {
public:
    
    template<typename T>
    inline static vec<T, 4> blend(vec<T, 4> src, vec<T, 4> dst, BlendParams params) {
        T src_f = T((0.2126 * src.r + 0.7152 * src.g + 0.0722 * src.b) >
                    (0.2126 * dst.r + 0.7152 * dst.g + 0.0722 * dst.b));
        return vec<T, 4>(src_f * src.rgb * dst[params.alpha_index] + (1.0 - src_f) * dst.rgb * src[params.alpha_index] +
                         src.rgb * (1.0 - dst[params.alpha_index]) + dst.rgb * (1.0 - src[params.alpha_index]),
                         src[params.alpha_index] + dst[params.alpha_index] * (1.0 - src[params.alpha_index]));
    }
};



class CBlendDarkerColor {
public:
    
    template<typename T>
    inline static vec<T, 4> blend(vec<T, 4> src, vec<T, 4> dst, BlendParams params) {
        T src_f = T((0.2126 * src.r + 0.7152 * src.g + 0.0722 * src.b) <=
                    (0.2126 * dst.r + 0.7152 * dst.g + 0.0722 * dst.b));
        return vec<T, 4>(src_f * src.rgb * dst[params.alpha_index] + (1.0 - src_f) * dst.rgb * src[params.alpha_index] +
                         src.rgb * (1.0 - dst[params.alpha_index]) + dst.rgb * (1.0 - src[params.alpha_index]),
                         src[params.alpha_index] + dst[params.alpha_index] * (1.0 - src[params.alpha_index]));
    }
};



class CBlendLuminosity {
public:
    
    template<typename T>
    inline static vec<T, 4> blend(vec<T, 4> src, vec<T, 4> dst, BlendParams params) {
        // Unpremultiplied src
        vec<T, 3> src_rgb = src.rgb / (src[params.alpha_index] + step(T(0.0), -src[params.alpha_index]));
        // unpremultiplied dst
        vec<T, 3> dst_rgb = dst.rgb / (dst[params.alpha_index] + step(T(0.0), -dst[params.alpha_index]));
        
        // SetLum
        vec<T, 3>  res_rgb = dst_rgb + dot(vec<T, 3>(0.3, 0.59, 0.11), src_rgb - dst_rgb);
        
        // Clip color
        T l = dot(vec<T, 3>(0.3, 0.59, 0.11), res_rgb);
        T n = min(min(res_rgb.r, res_rgb.g), res_rgb.b);
        T x = max(max(res_rgb.r, res_rgb.g), res_rgb.b);
        if (n < 0.0) {
            res_rgb = l + (res_rgb - l) *        l  / (l - n);
        }
        if (x > 1.0) {
            res_rgb = l + (res_rgb - l) * (1.0 - l) / (x - l);
        }
        
        // Without clamping additional artefacts are visible (black pixels in places where nothing should have been chagned).
        res_rgb = clamp(res_rgb, 0.0, 1.0);
        
        // Premultiplied
        return vec<T, 4>(((1.0 - src[params.alpha_index]) * dst.rgb) + ((1.0 - dst[params.alpha_index]) * src.rgb) + (src[params.alpha_index] * dst[params.alpha_index] * res_rgb),
                         src[params.alpha_index] + dst[params.alpha_index] * (1.0 - src[params.alpha_index]));
    }
};



class CBlendSaturate {
public:
    
    template<typename T>
    inline static vec<T, 4> blend(vec<T, 4> src, vec<T, 4> dst, BlendParams params) {
        // Unpremultiplied src
        vec<T, 3> src_rgb = src.rgb / (src[params.alpha_index] + step(T(0.0), -src[params.alpha_index]));
        // unpremultiplied dst
        vec<T, 3> dst_rgb = dst.rgb / (dst[params.alpha_index] + step(T(0.0), -dst[params.alpha_index]));
        
        // Sat(src)
        T s = max(max(src_rgb.r, src_rgb.g), src_rgb.b) - min(min(src_rgb.r, src_rgb.g), src_rgb.b);
        
        // Lum(dst)
        T l_dst = dot(vec<T, 3>(0.3, 0.59, 0.11), dst_rgb);
        
        bool3 clr_sort = dst_rgb.rgb > dst_rgb.brg;
        
        dst_rgb = mix(dst_rgb, vec<T, 3>(dst_rgb.y, dst_rgb.xz), step(dst_rgb.x, dst_rgb.y));
        dst_rgb = mix(dst_rgb, vec<T, 3>(dst_rgb.x, dst_rgb.zy), step(dst_rgb.y, dst_rgb.z));
        dst_rgb = mix(dst_rgb, vec<T, 3>(dst_rgb.y, dst_rgb.xz), step(dst_rgb.x, dst_rgb.y));
        
        dst_rgb = (dst_rgb.x == dst_rgb.z ?
                   vec<T, 3>(0.0) :
                   vec<T, 3>(s,
                             (dst_rgb.y - dst_rgb.z) * s / (dst_rgb.x - dst_rgb.z),
                             0.0));
        
        dst_rgb = mix(vec<T, 3>(dst_rgb.z, dst_rgb.yx), dst_rgb, T(clr_sort.x));
        dst_rgb = mix(dst_rgb, vec<T, 3>(dst_rgb.y, dst_rgb.xz), T(clr_sort.y == clr_sort.x));
        dst_rgb = mix(dst_rgb, vec<T, 3>(dst_rgb.x, dst_rgb.zy), T(clr_sort.z == clr_sort.x));
        
        // SetLum
        vec<T, 3>  res_rgb = dst_rgb + l_dst - dot(vec<T, 3>(0.3, 0.59, 0.11), dst_rgb);
        
        // ClipClr(res_rgb)
        T l = dot(vec<T, 3>(0.3, 0.59, 0.11), res_rgb);
        T n = min(min(res_rgb.r, res_rgb.g), res_rgb.b);
        T x = max(max(res_rgb.r, res_rgb.g), res_rgb.b);
        if (n < 0.0) {
            res_rgb = l + (res_rgb - l) *        l  / (l - n);
        }
        if (x > 1.0) {
            res_rgb = l + (res_rgb - l) * (1.0 - l) / (x - l);
        }
        
        
        // Premultiplied
        return vec<T, 4>(((1.0 - src[params.alpha_index]) * dst.rgb) + ((1.0 - dst[params.alpha_index]) * src.rgb) + (src[params.alpha_index] * dst[params.alpha_index] * res_rgb),
                         src[params.alpha_index] + dst[params.alpha_index] * (1.0 - src[params.alpha_index]));
    }
};



class CBlendHue {
public:
    
    template<typename T>
    inline static vec<T, 4> blend(vec<T, 4> src, vec<T, 4> dst, BlendParams params) {
        // Unpremultiplied src
        vec<T, 3> src_rgb = src.rgb / (src[params.alpha_index] + step(T(0.0), -src[params.alpha_index]));
        // unpremultiplied dst
        vec<T, 3> dst_rgb = dst.rgb / (dst[params.alpha_index] + step(T(0.0), -dst[params.alpha_index]));
        
        // Sat(src)
        T s = max(max(dst_rgb.r, dst_rgb.g), dst_rgb.b) - min(min(dst_rgb.r, dst_rgb.g), dst_rgb.b);
        
        // Lum(dst)
        T l_dst = dot(vec<T, 3>(0.3, 0.59, 0.11), dst_rgb);
        
        bool3 clr_sort = src_rgb.rgb > src_rgb.brg;
        
        src_rgb = mix(src_rgb, vec<T, 3>(src_rgb.y, src_rgb.xz), step(src_rgb.x, src_rgb.y));
        src_rgb = mix(src_rgb, vec<T, 3>(src_rgb.x, src_rgb.zy), step(src_rgb.y, src_rgb.z));
        src_rgb = mix(src_rgb, vec<T, 3>(src_rgb.y, src_rgb.xz), step(src_rgb.x, src_rgb.y));
        
        src_rgb = (src_rgb.x == src_rgb.z ?
                   vec<T, 3>(0.0) :
                   vec<T, 3>(s,
                             (src_rgb.y - src_rgb.z) * s / (src_rgb.x - src_rgb.z),
                             0.0));
        
        src_rgb = mix(vec<T, 3>(src_rgb.z, src_rgb.yx), src_rgb, T(clr_sort.x));
        src_rgb = mix(src_rgb, vec<T, 3>(src_rgb.y, src_rgb.xz), T(clr_sort.y == clr_sort.x));
        src_rgb = mix(src_rgb, vec<T, 3>(src_rgb.x, src_rgb.zy), T(clr_sort.z == clr_sort.x));
        
        // SetLum()
        vec<T, 3>  res_rgb = src_rgb + l_dst - dot(vec<T, 3>(0.3, 0.59, 0.11), src_rgb);
        
        // ClipClr(res_rgb)
        T l = dot(vec<T, 3>(0.3, 0.59, 0.11), res_rgb);
        T n = min(min(res_rgb.r, res_rgb.g), res_rgb.b);
        T x = max(max(res_rgb.r, res_rgb.g), res_rgb.b);
        if (n < 0.0) {
            res_rgb = l + (res_rgb - l) *        l  / (l - n);
        }
        if (x > 1.0) {
            res_rgb = l + (res_rgb - l) * (1.0 - l) / (x - l);
        }
        
        // Premultiplied
        return vec<T, 4>(((1.0 - src[params.alpha_index]) * dst.rgb) + ((1.0 - dst[params.alpha_index]) * src.rgb) + (src[params.alpha_index] * dst[params.alpha_index] * res_rgb),
                         src[params.alpha_index] + dst[params.alpha_index] * (1.0 - src[params.alpha_index]));
    }
};



class CBlendColor {
public:
    
    template<typename T>
    inline static vec<T, 4> blend(vec<T, 4> src, vec<T, 4> dst, BlendParams params) {
        // Unpremultiplied src
        vec<T, 3> src_rgb = src.rgb / (src[params.alpha_index] + step(T(0.0), -src[params.alpha_index]));
        // unpremultiplied dst
        vec<T, 3> dst_rgb = dst.rgb / (dst[params.alpha_index] + step(T(0.0), -dst[params.alpha_index]));
        
        // SetLum
        vec<T, 3>  res_rgb = src_rgb + dot(vec<T, 3>(0.3, 0.59, 0.11), dst_rgb - src_rgb);
        
        // ClipClr
        T l = dot(vec<T, 3>(0.3, 0.59, 0.11), res_rgb);
        T n = min(min(res_rgb.r, res_rgb.g), res_rgb.b);
        T x = max(max(res_rgb.r, res_rgb.g), res_rgb.b);
        if (n < 0.0) {
            res_rgb = l + (res_rgb - l) *        l  / (l - n);
        }
        if (x > 1.0) {
            res_rgb = l + (res_rgb - l) * (1.0 - l) / (x - l);
        }
        
        // Premultiplied
        return vec<T, 4>(((1.0 - src[params.alpha_index]) * dst.rgb) + ((1.0 - dst[params.alpha_index]) * src.rgb) + (src[params.alpha_index] * dst[params.alpha_index] * res_rgb),
                         src[params.alpha_index] + dst[params.alpha_index] * (1.0 - src[params.alpha_index]));
    }
};



class CBlendDivide {
public:
    
    template<typename T>
    inline static vec<T, 4> blend(vec<T, 4> src, vec<T, 4> dst, BlendParams params) {
        // Unpremultiplied src
        vec<T, 3> src_rgb = src.rgb / (src[params.alpha_index] + step(T(0.0), -src[params.alpha_index]));
        // unpremultiplied dst
        vec<T, 3> dst_rgb = dst.rgb / (dst[params.alpha_index] + step(T(0.0), -dst[params.alpha_index]));
        
        vec<T, 3> res_rgb = clamp(dst_rgb / src_rgb, T(0.0), T(1.0));
        
        // Premultiplied
        return vec<T, 4>(((1.0 - src[params.alpha_index]) * dst.rgb) + ((1.0 - dst[params.alpha_index]) * src.rgb) + (src[params.alpha_index] * dst[params.alpha_index] * res_rgb),
                         src[params.alpha_index] + dst[params.alpha_index] * (1.0 - src[params.alpha_index]));
    }
};



class CBlendSubtract {
public:
    
    template<typename T>
    inline static vec<T, 4> blend(vec<T, 4> src, vec<T, 4> dst, BlendParams params) {
        // Unpremultiplied src
        vec<T, 3> src_rgb = src.rgb / (src[params.alpha_index] + step(T(0.0), -src[params.alpha_index]));
        // unpremultiplied dst
        vec<T, 3> dst_rgb = dst.rgb / (dst[params.alpha_index] + step(T(0.0), -dst[params.alpha_index]));
        
        vec<T, 3> res_rgb = clamp(dst_rgb - src_rgb, T(0.0), T(1.0));
        
        // Premultiplied
        return vec<T, 4>(((1.0 - src[params.alpha_index]) * dst.rgb) + ((1.0 - dst[params.alpha_index]) * src.rgb) + (src[params.alpha_index] * dst[params.alpha_index] * res_rgb),
                         src[params.alpha_index] + dst[params.alpha_index] * (1.0 - src[params.alpha_index]));
    }
};



class CBlendDissolve {
public:
    
    template<typename T>
    inline static vec<T, 4> blend(vec<T, 4> src, vec<T, 4> dst, BlendParams params) {
        
        float2 xy = (params.cur_position + 1.0) * 0.5;
        
        float dy = fract(sin(dot(xy, float2(75.542, 35.7124))) * 22854.812);
        xy.y += dy;
        float randomVal = fract(sin(dot(xy, float2(2.124, 24.3598))) * 22854.812);
        
        float colorOpacity = src[params.alpha_index];
        src.rgb /= (src[params.alpha_index] + step(T(0.0), -src[params.alpha_index]));
        src[params.alpha_index] = step(T(0.0001), src[params.alpha_index]);
        src.rgb *= src[params.alpha_index];
        
        return src[params.alpha_index] == 0.0 ? dst : (randomVal < colorOpacity ? src : dst);
    }
};



class CBlendHardmix {
public:
    
    template<typename T>
    inline static vec<T, 4> blend(vec<T, 4> src, vec<T, 4> dst, BlendParams params) {
        // Unpremultiplied src
        vec<T, 3> src_rgb = src.rgb / (src[params.alpha_index] + step(T(0.0), -src[params.alpha_index]));
        // unpremultiplied dst
        vec<T, 3> dst_rgb = dst.rgb / (dst[params.alpha_index] + step(T(0.0), -dst[params.alpha_index]));
        
        vec<T, 3> res_rgb = step(T(1.0), src_rgb + dst_rgb);
        
        // Premultiplied
        return vec<T, 4>(((1.0 - src[params.alpha_index]) * dst.rgb) + ((1.0 - dst[params.alpha_index]) * src.rgb) + (src[params.alpha_index] * dst[params.alpha_index] * res_rgb),
                         src[params.alpha_index] + dst[params.alpha_index] * (1.0 - src[params.alpha_index]));
    }
};



class CBlendBehind {
public:
    
    template<typename T>
    inline static vec<T, 4> blend(vec<T, 4> src, vec<T, 4> dst, BlendParams params) {
        return dst + src * (1.0 - dst[params.alpha_index]);
    }
};


template<typename T>
vec<T, 4> blend(vec<T, 4> src, vec<T, 4> dst, int alphaIndex, float2 curPos, int blendType);

template<typename T>
vec<T, 4> blend(vec<T, 4> src, vec<T, 4> dst, int alphaIndex, float2 curPos, int blendType) {
    vec<T, 4> res = 0.0;
    BlendParams params = {
        alphaIndex,
        curPos
    };
    switch (blendType) {
        case BLEND_NORMAL:
            res = CBlendNormal::blend(src, dst, params);
            break;
        case BLEND_DARKEN:
            res = CBlendDarken::blend(src, dst, params);
            break;
        case BLEND_LIGHTEN:
            res = CBlendLighten::blend(src, dst, params);
            break;
        case BLEND_MULTIPLY:
            res = CBlendMultiply::blend(src, dst, params);
            break;
        case BLEND_SCREEN:
            res = CBlendScreen::blend(src, dst, params);
            break;
        case BLEND_COLORBURN:
            res = CBlendColorBurn::blend(src, dst, params);
            break;
        case BLEND_LINEARBURN:
            res = CBlendLinearBurn::blend(src, dst, params);
            break;
        case BLEND_COLORDODGE:
            res = CBlendColorDodge::blend(src, dst, params);
            break;
        case BLEND_LINEARDODGE:
            res = CBlendLinearDodge::blend(src, dst, params);
            break;
        case BLEND_OVERLAY:
            res = CBlendOverlay::blend(src, dst, params);
            break;
        case BLEND_SOFTLIGHT:
            res = CBlendSoftLight::blend(src, dst, params);
            break;
        case BLEND_HARDLIGHT:
            res = CBlendHardLight::blend(src, dst, params);
            break;
        case BLEND_VIVIDLIGHT:
            res = CBlendVividLight::blend(src, dst, params);
            break;
        case BLEND_LINEARLIGHT:
            res = CBlendLinearLight::blend(src, dst, params);
            break;
        case BLEND_PINLIGHT:
            res = CBlendPinLight::blend(src, dst, params);
            break;
        case BLEND_DIFFERENCE:
            res = CBlendDifference::blend(src, dst, params);
            break;
        case BLEND_EXCLUSION:
            res = CBlendExclusion::blend(src, dst, params);
            break;
        case BLEND_LIGHTERCOLOR:
            res = CBlendLighterColor::blend(src, dst, params);
            break;
        case BLEND_DARKERCOLOR:
            res = CBlendDarkerColor::blend(src, dst, params);
            break;
        case BLEND_LUMINOSITY:
            res = CBlendLuminosity::blend(src, dst, params);
            break;
        case BLEND_SATURATE:
            res = CBlendSaturate::blend(src, dst, params);
            break;
        case BLEND_HUE:
            res = CBlendHue::blend(src, dst, params);
            break;
        case BLEND_COLOR:
            res = CBlendColor::blend(src, dst, params);
            break;
        case BLEND_DIVIDE:
            res = CBlendDivide::blend(src, dst, params);
            break;
        case BLEND_SUBTRACT:
            res = CBlendSubtract::blend(src, dst, params);
            break;
        case BLEND_DISSOLVE:
            res = CBlendDissolve::blend(src, dst, params);
            break;
        case BLEND_HARDMIX:
            res = CBlendHardmix::blend(src, dst, params);
            break;
        case BLEND_BEHIND:
            res = CBlendBehind::blend(src, dst, params);
            break;
    };
    return res;
}



template<typename T>
vec<T, 4> blendCC(vec<T, 4> src, vec<T, 4> dst, int alphaIndex, float2 curPos, int blendType, bool isSRGB);

template<typename T>
vec<T, 4> blendCC(vec<T, 4> src, vec<T, 4> dst, int alphaIndex, float2 curPos, int blendType, bool isSRGB) {
    vec<T, 4> res = src;
    if (isSRGB) {
        res = blend(res, dst, alphaIndex, curPos, blendType);
    }
    else {
        T sa = res[alphaIndex];
        res /= sa + step(T(0.0), -sa);
        res *= res;
        res *= sa;
        dst *= dst;
        
        res = blend(res, dst, alphaIndex, curPos, blendType);
        
        res  = sqrt(res);
    }
    return res;
}



template<typename T>
vec<T, 4> blendCCPM(vec<T, 4> src, vec<T, 4> dst, bool isSRGB, T alpha);

template<typename T>
vec<T, 4> blendCCPM(vec<T, 4> src, vec<T, 4> dst, bool isSRGB, T alpha) {
    vec<T, 4> res = src;
    if (isSRGB) {
        res = mix(dst, res, alpha);
    }
    else {
        res *= res;
        dst *= dst;
        
        res = mix(dst, res, alpha);
        
        res  = sqrt(res);
    }
    return res;
}
